/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa;

import org.kuali.rice.core.api.criteria.GenericQueryResults;
import org.kuali.rice.core.api.criteria.QueryByCriteria;

/**
 * Simple interface that abstracts ORM query by criteria and delete logic.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
interface CriteriaQuery {
    /**
     * Looks up a type based on a query criteria.
     *
     * @param queryClass the class to lookup
     * @param criteria the criteria to lookup against. cannot be null.
     * @param <T> the type that is being looked up.
     * @return the results. will never be null.
     * @throws IllegalArgumentException if the criteria is null
     */
    <T> GenericQueryResults<T> lookup(final Class<T> queryClass, final QueryByCriteria criteria);

    /**
     * Deletes all data objects based on the given type.
     *
     * @param type the type of data objects to delete
     * @param <T> the data object class type
     *
     * @throws IllegalArgumentException if the class type is null
     * @throws org.springframework.dao.DataAccessException if data access fails
     */
    public <T> void deleteAll(Class<T> type);


    /**
     * Deletes data objects based on the given criteria
     *
     * <p>If the given criteria is empty or null than an {@link java.lang.IllegalArgumentException} will be thrown.
     *   If the given type is null then an {@link java.lang.IllegalArgumentException} will be thrown.</p>
     *
     * @param type the type of data object
     * @param criteria criteria to filter by
     *
     * @throws IllegalArgumentException if the criteria or criteria predicate is null
     * @throws org.springframework.dao.DataAccessException if data access fails
     */
    public <T> void deleteMatching(Class<T> type, QueryByCriteria criteria);

}
