/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.document;

import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.krad.bo.DataObjectAuthorizer;

/**
 * Authorizer class for {@link Document} instances
 *
 * <p>
 * Authorizer provides user based authorization
 * </p>
 *
 * <p>
 * The document authorizer is associated with a document type through its data dictionary
 * {@link org.kuali.rice.krad.datadictionary.DocumentEntry}. This is then used by the framework to authorize certain
 * actions and in addition used for view presentation logic
 * </p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface DocumentAuthorizer extends DataObjectAuthorizer {
    
    /**
     * Checks if a user has the permissions to initiate a  document
     *
     * @param documentTypeName document type name
     * @param user current user
     * @return boolean, true if the user has the permissions to initiate a document else false
     */

    public boolean canInitiate(String documentTypeName, Person user);

    /**
     * Checks if a user has the permissions to open a  document
     *
     * @param document document to check
     * @param user current user
     * @return boolean, true if the user has the permissions to open a document else false
     */

    public boolean canOpen(Document document, Person user);

    /**
     * Determines if the document can be edited; if false is returned, then all fields are in a
     * read only state
     *
     * @param document document to check
     * @param user current user
     * @return boolean, true if the user has the permissions to edit a document else false
     */

    public boolean canEdit(Document document, Person user);

    public boolean canAnnotate(Document document, Person user);

    public boolean canReload(Document document, Person user);

    public boolean canClose(Document document, Person user);

    public boolean canSave(Document document, Person user);

    /**
     * Determines if the user has permission to route the document
     *
     * @param document document to check
     * @param user current user
     * @return boolean, true if the user has permissions to route a document else false
     */
    public boolean canRoute(Document document, Person user);

    /**
     * Determines if the user has permission to cancel the document
     *
     * @param document document to check
     * @param user current user
     * @return boolean, true if the user has permissions to cancel a document else false
     */
    public boolean canCancel(Document document, Person user);

    /**
     * Determines if the user has permission to copy the document
     *
     * @param document document to check
     * @param user current user
     * @return boolean, true if the user has permissions to cancel a document else false
     */
    public boolean canCopy(Document document, Person user);

    public boolean canPerformRouteReport(Document document, Person user);

    public boolean canBlanketApprove(Document document, Person user);

    public boolean canApprove(Document document, Person user);

    public boolean canDisapprove(Document document, Person user);

    public boolean canSendNoteFyi(Document document, Person user);

    public boolean canEditDocumentOverview(Document document, Person user);

    public boolean canFyi(Document document, Person user);

    public boolean canAcknowledge(Document document, Person user);

    public boolean canReceiveAdHoc(Document document, Person user, String actionRequestCode);

    public boolean canAddNoteAttachment(Document document, String attachmentTypeCode, Person user);

    public boolean canDeleteNoteAttachment(Document document, String attachmentTypeCode,
            String authorUniversalIdentifier, Person user);

    public boolean canViewNoteAttachment(Document document, String attachmentTypeCode, Person user);

    @Deprecated
    public boolean canViewNoteAttachment(Document document, String attachmentTypeCode, String authorUniversalIdentifier,
            Person user);

    public boolean canSendAdHocRequests(Document document, String actionRequestCd, Person user);

    public boolean canSendAnyTypeAdHocRequests(Document document, Person user);

    public boolean canTakeRequestedAction(Document document, String actionRequestCode, Person user);

    /**
     * @since 2.1
     */
    public boolean canRecall(Document document, Person user);

    /**
     * Determines if the user has permission to take a super user action.
     *
     * @param document document to check
     * @param user current user
     *
     * @return true if the user has permissions to take a super user action, otherwise false
     *
     * @since 2.5
     */
    boolean canSuperUserTakeAction(Document document, Person user);

    /**
     * Determines if the user has permission to approve a document as a super user.
     *
     * @param document document to check
     * @param user current user
     *
     * @return true if the user has permissions to approve a document as a super user, otherwise false
     * @since 2.5
     */
    boolean canSuperUserApprove(Document document, Person user);

    /**
     * Determines if the user has permission to disapprove a document as a super user.
     *
     * @param document document to check
     * @param user current user
     *
     * @return true if the user has permissions to disapprove a document as a super user, otherwise false
     * @since 2.5
     */
    boolean canSuperUserDisapprove(Document document, Person user);

    void setDocumentRequestAuthorizationCache(DocumentRequestAuthorizationCache documentRequestAuthorizationCache);

}
