/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.web.bind;

import org.kuali.rice.core.api.util.RiceKeyConstants;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.core.api.util.type.KualiPercent;
import org.kuali.rice.core.web.format.FormatException;

import java.beans.PropertyEditorSupport;
import java.io.Serializable;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;

/**
 * PropertyEditor converts between percentage strings and
 * <code>org.kuali.rice.core.api.util.type.KualiPercent</code> objects
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class UifPercentageEditor extends PropertyEditorSupport implements Serializable {
    private static final long serialVersionUID = -3562156375311932094L;

    /** The default scale for percentage values. */
    public final static int PERCENTAGE_SCALE = 2;

    /** The default format for percentage values. */
    public final static String PERCENTAGE_FORMAT = "#,##0.00";

    /**
     * This overridden method converts
     * <code>org.kuali.rice.core.api.util.type.KualiPercent</code> objects to the
     * display string.
     *
     * @see java.beans.PropertyEditorSupport#getAsText()
     */
    @Override
    public String getAsText() {
        Object value = this.getValue();
        // Previously returned N/A
        if (value == null)
            return "";

        String stringValue = "";
        try {
            if (value instanceof KualiDecimal) {
                value = ((KualiDecimal) this.getValue()).bigDecimalValue();
            }
            BigDecimal bigDecValue = (BigDecimal) value;
            bigDecValue = bigDecValue.setScale(PERCENTAGE_SCALE, BigDecimal.ROUND_HALF_UP);
            stringValue = NumberFormat.getInstance().format(bigDecValue.doubleValue());
        } catch (IllegalArgumentException iae) {
            throw new FormatException("formatting", RiceKeyConstants.ERROR_PERCENTAGE, this.getValue().toString(), iae);
        }

        return stringValue;
    }

    /**
     * This overridden method converts the display string to a
     * <code>org.kuali.rice.core.api.util.type.KualiPercent</code> object.
     *
     * @see java.beans.PropertyEditorSupport#setAsText(java.lang.String)
     */
    @Override
    public void setAsText(String text) throws IllegalArgumentException {
        try {
            DecimalFormat formatter = new DecimalFormat(PERCENTAGE_FORMAT);
            Number parsedNumber = formatter.parse(text.trim());
            this.setValue(new KualiPercent(parsedNumber.doubleValue()));
        } catch (NumberFormatException e) {
            throw new FormatException("parsing", RiceKeyConstants.ERROR_PERCENTAGE, text, e);
        } catch (ParseException e) {
            throw new FormatException("parsing", RiceKeyConstants.ERROR_PERCENTAGE, text, e);
        }
    }

}
