/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krms.impl.repository;

import org.kuali.rice.core.api.uif.RemotableAttributeField;
import org.kuali.rice.kns.inquiry.KualiInquirableImpl;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.service.KRADServiceLocator;
import org.kuali.rice.krad.uif.container.Container;
import org.kuali.rice.krad.uif.view.View;
import org.kuali.rice.krad.web.form.InquiryForm;
import org.kuali.rice.krms.impl.ui.AgendaEditor;
import org.kuali.rice.krms.impl.util.KrmsRetriever;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class AgendaInquiryHelperServiceImpl extends KualiInquirableImpl {

    private transient KrmsRetriever krmsRetriever = new KrmsRetriever();

    private DataObjectService dataObjectService;

    @Override
    public AgendaEditor retrieveDataObject(Map fieldValues) {
        AgendaEditor agendaEditor = null;

        AgendaBo agenda = getDataObjectService().find(AgendaBo.class, fieldValues.get("id"));
        if (agenda != null) {
            agendaEditor = new AgendaEditor();
            agendaEditor.setAgenda(agenda);
            agendaEditor.setNamespace(agenda.getContext().getNamespace());
            agendaEditor.setContextName(agenda.getContext().getName());
            agendaEditor.setCustomAttributesMap(agenda.getAttributes());
        }

        return agendaEditor;
    }

    /**
     * Returns the AgendaEditor from the given InquiryForm
     *
     * @param model InquiryFrom to retrieve the AgendaEditor from.
     * @return AgendaEditor retrieved from the given InquiryForm.
     */
    private AgendaEditor retrieveAgendaEditor(InquiryForm model) {
        InquiryForm inquiryForm = (InquiryForm) model;
        return (AgendaEditor) inquiryForm.getDataObject();
    }

    /**
     * Returns the Agenda's RemotableAttributeFields
     *
     * @param model InquiryFrom to retrieve the AgendaEditor from.
     * @return List&lt;RemotableAttributeField&gt;
     */
    public List<RemotableAttributeField> retrieveAgendaCustomAttributes(View view, Object model, Container container) {
        AgendaEditor agendaEditor = retrieveAgendaEditor((InquiryForm) model);
        return krmsRetriever.retrieveAgendaCustomAttributes(agendaEditor);
    }

    /**
     * Returns the Rule Action RemotableAttributeFields. This only supports a single action within a rule.
     *
     * @param model InquiryFrom to retrieve the AgendaEditor from.
     * @return List&lt;RemotableAttributeField&gt;
     */
    public List<RemotableAttributeField> retrieveRuleActionCustomAttributes(View view, Object model,
            Container container) {
        AgendaEditor agendaEditor = retrieveAgendaEditor((InquiryForm) model);
        return krmsRetriever.retrieveRuleActionCustomAttributes(agendaEditor);
    }

    /**
     * Returns the Rule RemotableAttributeFields. This only supports a single action within a rule.
     *
     * @param model InquiryFrom to retrieve the AgendaEditor from.
     * @return List&lt;RemotableAttributeField&gt;
     */
    public List<RemotableAttributeField> retrieveRuleCustomAttributes(View view, Object model, Container container) {
        AgendaEditor agendaEditor = retrieveAgendaEditor((InquiryForm) model);
        return krmsRetriever.retrieveRuleCustomAttributes(agendaEditor);
    }

    /**
     * Retrieve a list of {@link RemotableAttributeField}s for the parameters (if any) required by the resolver for
     * the selected term in the proposition that is under edit.  Since this method is part of the inquiry view,
     * non of the propositions will ever be under edit when it is called, and an empty list will be returned.
     *
     * @param model InquiryFrom to retrieve the AgendaEditor from.
     * @return List&lt;RemotableAttributeField&gt; Collections.emptyList()
     */
    public List<RemotableAttributeField> retrieveTermParameters(View view, Object model, Container container) {
        return Collections.emptyList();
    }

    @Override
    public DataObjectService getDataObjectService() {
        if (dataObjectService == null) {
            return KRADServiceLocator.getDataObjectService();
        }
        return dataObjectService;
    }

    @Override
    public void setDataObjectService(DataObjectService dataObjectService) {
        this.dataObjectService = dataObjectService;
    }

}
