/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krms.impl.rule;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.criteria.QueryByCriteria;
import org.kuali.rice.core.api.criteria.QueryResults;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.maintenance.MaintenanceDocument;
import org.kuali.rice.krad.rules.MaintenanceDocumentRuleBase;
import org.kuali.rice.krad.service.KRADServiceLocator;
import org.kuali.rice.krms.api.repository.term.TermDefinition;
import org.kuali.rice.krms.api.repository.term.TermSpecificationDefinition;
import org.kuali.rice.krms.impl.repository.KrmsRepositoryServiceLocator;
import org.kuali.rice.krms.impl.repository.TermBo;
import org.kuali.rice.krms.impl.repository.TermBoService;
import org.kuali.rice.krms.impl.util.KRMSPropertyConstants;

import java.util.HashMap;
import java.util.Map;

public class TermBusRule extends MaintenanceDocumentRuleBase {

    private DataObjectService dataObjectService;

    @Override
    protected boolean processCustomSaveDocumentBusinessRules(MaintenanceDocument document) {
        boolean isValid = true;

        TermBo term = (TermBo) document.getNewMaintainableObject().getDataObject();
        isValid &= validateId(term);
        isValid &= validateDescriptionNamespace(term);
        isValid &= validateTermSpecId(term);

        return isValid;
    }

    private boolean validateTermSpecId(TermBo term) {
        if (StringUtils.isBlank(term.getSpecificationId())) {
            this.putFieldError(KRMSPropertyConstants.Term.TERM_SPECIFICATION_ID, "error.term.invalidTermSpecification");

            return false;
        }

        TermSpecificationDefinition termSpec =
                KrmsRepositoryServiceLocator.getTermBoService().getTermSpecificationById(term.getSpecificationId());

        if (termSpec == null) {
            this.putFieldError(KRMSPropertyConstants.Term.TERM_SPECIFICATION_ID, "error.term.invalidTermSpecification");

            return false;
        }

        return true;
    }

    private boolean validateId(TermBo term) {
        if (StringUtils.isNotBlank(term.getId())) {
            TermDefinition termInDatabase = getTermBoService().getTerm(term.getId());
            if ((termInDatabase  != null) && (!StringUtils.equals(termInDatabase.getId(), term.getId()))) {
                this.putFieldError(KRMSPropertyConstants.Term.TERM_ID, "error.term.duplicateId");

                return false;
            }
        }

        return true;
    }

    /**
     * Check if the name-namespace pair already exist.
     * @param term
     * @return true if the name-namespace pair is unique, false otherwise
     */
    private boolean validateDescriptionNamespace(TermBo term) {
        if (term.getSpecification() != null && StringUtils.isNotBlank(term.getDescription()) && StringUtils.isNotBlank(
                term.getSpecification().getNamespace())) {

            Map<String, String> critMap = new HashMap<String, String>();

            critMap.put("description", term.getDescription());
            critMap.put("specification.namespace", term.getSpecification().getNamespace());
            QueryByCriteria criteria = QueryByCriteria.Builder.andAttributes(critMap).build();
            QueryResults<TermBo> queryResults = getDataObjectService().findMatching(TermBo.class, criteria);

            TermBo termInDatabase = null;

            if (!CollectionUtils.isEmpty(queryResults.getResults()) && queryResults.getResults().size() == 1) {
                termInDatabase = queryResults.getResults().get(0);
            }

            if((termInDatabase != null) && (!StringUtils.equals(termInDatabase.getId(), term.getId()))) {
                this.putFieldError(KRMSPropertyConstants.Term.DESCRIPTION, "error.term.duplicateNameNamespace");

                return false;
            }
        }

        return true;
    }

    public TermBoService getTermBoService() {
        return KrmsRepositoryServiceLocator.getTermBoService();
    }

    @Override
    public DataObjectService getDataObjectService() {
        if(dataObjectService == null){
            return KRADServiceLocator.getDataObjectService();
        }

        return dataObjectService;
    }

    public void setBoService(DataObjectService dataObjectService) {
        this.dataObjectService = dataObjectService;
    }

}
