/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging;

import java.lang.reflect.Method;
import java.sql.Timestamp;

import javax.xml.namespace.QName;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.config.property.Config;
import org.kuali.rice.core.api.config.property.ConfigContext;
import org.kuali.rice.core.api.exception.RiceRuntimeException;
import org.kuali.rice.ksb.api.KsbApiServiceLocator;
import org.kuali.rice.ksb.api.bus.Endpoint;
import org.kuali.rice.ksb.api.bus.ServiceBus;
import org.kuali.rice.ksb.api.bus.ServiceConfiguration;
import org.kuali.rice.ksb.api.messaging.AsynchronousCall;
import org.kuali.rice.ksb.service.KSBServiceLocator;
import org.kuali.rice.ksb.util.KSBConstants;

/**
 * Handles invocation of a {@link PersistedMessageBO}.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class MessageServiceInvoker implements Runnable {

    protected static final Logger LOG = LogManager.getLogger(MessageServiceInvoker.class);

    private PersistedMessageBO message;

    public MessageServiceInvoker(PersistedMessageBO message) {
        this.message = message;
    }

    @Override
    public void run() {
        LOG.debug("calling service from persisted message " + getMessage().getRouteQueueId());
        if(ConfigContext.getCurrentContextConfig().getBooleanProperty(Config.MESSAGE_PERSISTENCE)) {
            PersistedMessageBO messageFromDB = KSBServiceLocator.getMessageQueueService().findByRouteQueueId(getMessage().getRouteQueueId());
            if(messageFromDB == null) {
                LOG.warn("On message invocation, message payload not found for message " + getMessage());
                return;
            }
        }

        try {
            KSBServiceLocator.getTransactionTemplate().execute(status -> {
                AsynchronousCall methodCall = getMessage().getMethodCall();
                Object result1;
                try {
                    result1 = invokeService(methodCall);
                    KSBServiceLocator.getMessageQueueService().delete(getMessage());
                } catch (Throwable t) {
                    LOG.warn("Caught throwable making async service call " + methodCall, t);
                    throw new MessageProcessingException(t);
                }
                return result1;
            });
        } catch (Throwable t) {
        	// if we are in synchronous mode, we can't put the message into exception routing, let's instead throw the error up to the calling code
        	// however, for the purposes of the unit tests, even when in synchronous mode, we want to call the exception routing service, so check a parameter for that as well
        	boolean allowSyncExceptionRouting = Boolean.parseBoolean(ConfigContext.getCurrentContextConfig().getProperty(KSBConstants.Config.ALLOW_SYNC_EXCEPTION_ROUTING));
     	 	if (!allowSyncExceptionRouting && KSBConstants.MESSAGING_SYNCHRONOUS.equals(ConfigContext.getCurrentContextConfig().getProperty(KSBConstants.Config.MESSAGE_DELIVERY))) {
     	 		if (t instanceof RuntimeException) {
     	 			throw (RuntimeException)t;
     	 		}
     	 		throw new RiceRuntimeException(t);
     	 	} else {
     	 		placeInExceptionRouting(t);
     	 	}
        }
    }

    /**
     * Executed when an exception is encountered during message invocation.
     * Attempts to call the ExceptionHandler for the message, if that fails it
     * will attempt to set the status of the message in the queue to
     * "EXCEPTION".
     */
    protected void placeInExceptionRouting(Throwable t) {
        LOG.error("Error processing message: " + this.message, t);
        final Throwable throwable;
        if (t instanceof MessageProcessingException) {
            throwable = t.getCause();
        } else {
            throwable = t;
        }
        try {
        	try {
        		KSBServiceLocator.getExceptionRoutingService().placeInExceptionRouting(throwable, this.message);
        	} catch (Throwable t1) {
        		KSBServiceLocator.getExceptionRoutingService().placeInExceptionRoutingLastDitchEffort(throwable, this.message);
        	}
        } catch (Throwable t2) {
            LOG.error("An error was encountered when invoking exception handler for message. Attempting to change message status to EXCEPTION.", t2);
            message.setQueueStatus(KSBConstants.ROUTE_QUEUE_EXCEPTION);
            message.setQueueDate(new Timestamp(System.currentTimeMillis()));
            try {
                message = KSBServiceLocator.getMessageQueueService().save(message);
            } catch (Throwable t3) {
                LOG.fatal("Failed to flip status of message to EXCEPTION!!!", t3);
            }
        }
    }

    /**
     * Invokes the AsynchronousCall represented on the methodCall on the service
     * contained in the ServiceInfo object on the AsynchronousCall.
     * 
     */
    protected Object invokeService(AsynchronousCall methodCall) throws Exception {
        ServiceConfiguration serviceConfiguration = methodCall.getServiceConfiguration();
        QName serviceName = serviceConfiguration.getServiceName();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Attempting to call service " + serviceName);
        }

        Object service = getService(serviceConfiguration);
        if (service == null) {
        	throw new RiceRuntimeException("Failed to locate service endpoint for message: " + serviceConfiguration);
        }
        Method method = service.getClass().getMethod(methodCall.getMethodName(), methodCall.getParamTypes());
        return method.invoke(service, methodCall.getArguments());
    }

    protected Object getService(ServiceConfiguration serviceConfiguration) {
        Object service;
        if (serviceConfiguration.isQueue()) {
            service = getQueueService(serviceConfiguration);
        } else {
            service = getTopicService(serviceConfiguration);
        }
        return service;
    }

    /**
     * Get the service as a topic. This means we want to contact every service
     * that is a part of this topic. We've grabbed all the services that are a
     * part of this topic and we want to make sure that we get everyone of them =
     * that is we want to circumvent loadbalancing and therefore not ask for the
     * service by it's name but the url to get the exact service we want.
     */
    protected Object getTopicService(ServiceConfiguration serviceConfiguration) {
        // get the service locally if we have it so we don't go through any
        // remoting
        ServiceBus serviceBus = KsbApiServiceLocator.getServiceBus();
        Endpoint endpoint = serviceBus.getConfiguredEndpoint(serviceConfiguration);
        if (endpoint == null) {
        	return null;
        }
        return endpoint.getService();
    }

    /**
     * Because this is a queue we just need to grab one.
     */
    protected Object getQueueService(ServiceConfiguration serviceConfiguration) {
    	ServiceBus serviceBus = KsbApiServiceLocator.getServiceBus();
    	return serviceBus.getService(serviceConfiguration.getServiceName(), serviceConfiguration.getApplicationId());
    }

    public PersistedMessageBO getMessage() {
        return message;
    }
}
