/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.security;

import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.Signature;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;

/**
 * An InputStream which decorates another InputStream with a wrapper that verifies the digital signature
 * of the data after the last piece of data is read.  The digital signature to verify against is
 * passed into the constructor of this stream.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class SignatureVerifyingInputStream extends ServletInputStream {

	private final byte[] digitalSignature;
	private final Signature signature;
	private final InputStream wrappedInputStream;
	private boolean finished = false;
	
	public SignatureVerifyingInputStream(byte[] digitalSignature, Signature signature, InputStream wrappedInputStream) {
		this.digitalSignature = digitalSignature;
		this.signature = signature;
		this.wrappedInputStream = wrappedInputStream;
	}

	@Override
	public synchronized int read() throws IOException {
		int data = this.wrappedInputStream.read();
		try {
			if (data == -1) {
				finished = true;
				verifySignature();
			} else {
			    this.signature.update((byte)data);
			}
		} catch (GeneralSecurityException e) {
			throw new IOException("Error processing digital signature.", e);
		}
		return data;
	}
	
	protected void verifySignature() throws IOException, GeneralSecurityException {
		boolean verifies = this.signature.verify(this.digitalSignature);
		if (!verifies) {
			throw new IOException("The digital signature could not be successfully verified!");
		}
	}

	@Override
	public boolean isFinished() {
		return finished;
	}

	@Override
	public boolean isReady() {
		return true;
	}

	public void setReadListener(ReadListener readListener) {
		throw new UnsupportedOperationException();
	}

}
