/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.security;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.GeneralSecurityException;

import javax.servlet.ServletOutputStream;
import javax.servlet.WriteListener;

/**
 * An OutputStream which decorates another OutputStream with a wrapper that digitally
 * signs the data when the OutputStream is closed.  Since this class does not know where
 * the resulting digital signature will reside, a DigitalSigner will be invoked to
 * execute the actual signing of the message (i.e. put it in a header).
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class SignatureSigningOutputStream extends ServletOutputStream {

	private final boolean delayWrite;
	private final DigitalSigner signer;
	private BufferedOutputStream bufferedDataHoldingStream;
	private ByteArrayOutputStream dataHoldingStream;
	private final OutputStream wrappedOutputStream;
	
	/**
	 * Constructs a SignatureSigningOutputStream with the given DigitalSigner and underlying OutputStream.
	 * If true, the delayWrite boolean indicates that the stream should store all data internally until the
	 * stream is closed, at which point it should forward all data to the wrapped OutputStream.  If delayWrite
	 * is false, then the data will be forwarded immediately.
	 */
	public SignatureSigningOutputStream(DigitalSigner signer, OutputStream wrappedOutputStream, boolean delayWrite) {
		super();
		this.delayWrite = delayWrite;
		if (delayWrite) {
		    this.dataHoldingStream = new ByteArrayOutputStream();
		    this.bufferedDataHoldingStream = new BufferedOutputStream(this.dataHoldingStream);
		}
		this.wrappedOutputStream = wrappedOutputStream;
		this.signer = signer;
	}

	@Override
    public void write(int data) throws IOException {
		if (this.delayWrite) {
		    this.bufferedDataHoldingStream.write(data);
		} else {
		    this.wrappedOutputStream.write(data);
		}
		try {
		    this.signer.getSignature().update((byte)data);	
		} catch (GeneralSecurityException e) {
			throw new IOException("Error updating signature.", e);
		}
	}
	
	@Override
	public void close() throws IOException {
		// before we close, sign the message
		try {
		    this.signer.sign();
			if (this.delayWrite) {
			    this.bufferedDataHoldingStream.close();
				byte[] data = this.dataHoldingStream.toByteArray();
				for (byte datum : data) {
					this.wrappedOutputStream.write(datum);
				}
			}
			this.wrappedOutputStream.close();
		} catch (Exception e) {
			throw new IOException("Error attaching digital signature to outbound response.", e);
		} finally {
			super.close();
		}
	}

	@Override
	public boolean isReady() {
		return true;
	}

	@Override
	public void setWriteListener(WriteListener writeListener) {
		throw new UnsupportedOperationException();
	}
}
