/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.impl.registry;

import org.kuali.rice.krad.data.jpa.PortableSequenceGenerator;
import org.kuali.rice.ksb.api.registry.ServiceEndpointStatus;
import org.kuali.rice.ksb.api.registry.ServiceInfo;
import org.kuali.rice.ksb.api.registry.ServiceInfoContract;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Table;
import javax.xml.namespace.QName;
import java.io.Serializable;

/**
 * Model bean that represents the definition of a service on the bus.
 * 
 * @see ServiceInfo
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */

@Entity
@Table(name="KRSB_SVC_DEF_T")
public class ServiceInfoBo implements ServiceInfoContract, Serializable {
 
	private static final long serialVersionUID = -4244884858494208070L;

    @Id
    @GeneratedValue(generator = "KRSB_SVC_DEF_S")
    @PortableSequenceGenerator(name = "KRSB_SVC_DEF_S")
	@Column(name = "SVC_DEF_ID")
	private String serviceId;
	
	@Column(name = "SVC_NM")
	private String serviceName;
    
	@Column(name = "SVC_URL", length = 500)
	private String endpointUrl;
	
	@Column(name = "INSTN_ID")
	private String instanceId;

	@Column(name = "APPL_ID")
	private String applicationId;

	@Column(name = "SRVR_IP")
	private String serverIpAddress;
	
	@Column(name = "TYP_CD")
	private String type;
	
	@Column(name = "SVC_VER")
	private String serviceVersion;
	
	@Column(name = "STAT_CD")
	private String statusCode;
	
	@Column(name = "SVC_DSCRPTR_ID")
	private String serviceDescriptorId;
	
	@Column(name = "CHKSM", length = 30)
	private String checksum;

    @Deprecated
	@Column(name = "VER_NBR")
	private Long versionNumber;

    public ServiceInfoBo() {
        this.versionNumber = Long.valueOf(1);
    }

    @Override
    public String getServiceId() {
        return serviceId;
    }

    public void setServiceId(String serviceId) {
        this.serviceId = serviceId;
    }

    @Override
    public QName getServiceName() {
        if (this.serviceName == null) {
            return null;
        }
        return QName.valueOf(this.serviceName);
    }

    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    @Override
    public String getEndpointUrl() {
        return endpointUrl;
    }

    public void setEndpointUrl(String endpointUrl) {
        this.endpointUrl = endpointUrl;
    }

    @Override
    public String getInstanceId() {
        return instanceId;
    }

    public void setInstanceId(String instanceId) {
        this.instanceId = instanceId;
    }

    @Override
    public String getApplicationId() {
        return applicationId;
    }

    public void setApplicationId(String applicationId) {
        this.applicationId = applicationId;
    }

    @Override
    public String getServerIpAddress() {
        return serverIpAddress;
    }

    public void setServerIpAddress(String serverIpAddress) {
        this.serverIpAddress = serverIpAddress;
    }

    @Override
    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    @Override
    public String getServiceVersion() {
        return serviceVersion;
    }

    public void setServiceVersion(String serviceVersion) {
        this.serviceVersion = serviceVersion;
    }

    public String getStatusCode() {
        return statusCode;
    }

    public void setStatusCode(String statusCode) {
        this.statusCode = statusCode;
    }

    @Override
    public String getServiceDescriptorId() {
        return serviceDescriptorId;
    }

    public void setServiceDescriptorId(String serviceDescriptorId) {
        this.serviceDescriptorId = serviceDescriptorId;
    }

    @Override
    public String getChecksum() {
        return checksum;
    }

    public void setChecksum(String checksum) {
        this.checksum = checksum;
    }

    @Override
    public Long getVersionNumber() {
        return versionNumber;
    }

    /**
     * Version number is deprecated, so this method does nothing.
     *
     * @deprecated version number is no longer used
     */
    @Deprecated
    public void setVersionNumber(Long versionNumber) {
        // no longer does anything
    }

    @Override
	public ServiceEndpointStatus getStatus() {
		if (getStatusCode() == null) {
			return null;
		}
		return ServiceEndpointStatus.fromCode(getStatusCode());
	}

	public static ServiceInfo to(ServiceInfoBo bo) {
		if (bo == null) {
			return null;
		}
		return ServiceInfo.Builder.create(bo).build();
	}
	
	public static ServiceInfoBo from(ServiceInfo im) {
		if (im == null) {
			return null;
		}
		ServiceInfoBo bo = new ServiceInfoBo();
		bo.serviceId = im.getServiceId();
		bo.serviceName = im.getServiceName().toString();
		bo.endpointUrl = im.getEndpointUrl();
		bo.instanceId = im.getInstanceId();
		bo.applicationId = im.getApplicationId();
		bo.serverIpAddress = im.getServerIpAddress();
		bo.type = im.getType();
		bo.serviceVersion = im.getServiceVersion();
		bo.statusCode = im.getStatus().getCode();
		bo.serviceDescriptorId = im.getServiceDescriptorId();
		bo.checksum = im.getChecksum();
		bo.versionNumber = im.getVersionNumber();
		return bo;
	}
	
	
	
	
}
